/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.retry;

import java.util.ArrayDeque;
import org.jspecify.annotations.Nullable;
import org.springframework.core.log.LogAccessor;
import org.springframework.core.retry.RetryException;
import org.springframework.core.retry.RetryListener;
import org.springframework.core.retry.RetryOperations;
import org.springframework.core.retry.RetryPolicy;
import org.springframework.core.retry.Retryable;
import org.springframework.util.Assert;
import org.springframework.util.backoff.BackOffExecution;

public class RetryTemplate
implements RetryOperations {
    private static final LogAccessor logger = new LogAccessor(RetryTemplate.class);
    private RetryPolicy retryPolicy = RetryPolicy.withDefaults();
    private RetryListener retryListener = new RetryListener(){};

    public RetryTemplate() {
    }

    public RetryTemplate(RetryPolicy retryPolicy) {
        Assert.notNull((Object)retryPolicy, "RetryPolicy must not be null");
        this.retryPolicy = retryPolicy;
    }

    public void setRetryPolicy(RetryPolicy retryPolicy) {
        Assert.notNull((Object)retryPolicy, "Retry policy must not be null");
        this.retryPolicy = retryPolicy;
    }

    public RetryPolicy getRetryPolicy() {
        return this.retryPolicy;
    }

    public void setRetryListener(RetryListener retryListener) {
        Assert.notNull((Object)retryListener, "Retry listener must not be null");
        this.retryListener = retryListener;
    }

    public RetryListener getRetryListener() {
        return this.retryListener;
    }

    @Override
    public <R> @Nullable R execute(Retryable<? extends @Nullable R> retryable) throws RetryException {
        String retryableName = retryable.getName();
        try {
            logger.debug(() -> "Preparing to execute retryable operation '%s'".formatted(retryableName));
            R result = retryable.execute();
            logger.debug(() -> "Retryable operation '%s' completed successfully".formatted(retryableName));
            return result;
        }
        catch (Throwable initialException) {
            logger.debug(initialException, () -> "Execution of retryable operation '%s' failed; initiating the retry process".formatted(retryableName));
            BackOffExecution backOffExecution = this.retryPolicy.getBackOff().start();
            ArrayDeque<Throwable> exceptions = new ArrayDeque<Throwable>(4);
            exceptions.add(initialException);
            Throwable lastException = initialException;
            while (this.retryPolicy.shouldRetry(lastException)) {
                try {
                    long duration = backOffExecution.nextBackOff();
                    if (duration == -1L) break;
                    logger.debug(() -> "Backing off for %dms after retryable operation '%s'".formatted(duration, retryableName));
                    Thread.sleep(duration);
                }
                catch (InterruptedException interruptedException) {
                    Thread.currentThread().interrupt();
                    RetryInterruptedException retryException = new RetryInterruptedException("Unable to back off for retryable operation '%s'".formatted(retryableName), interruptedException);
                    exceptions.forEach(retryException::addSuppressed);
                    this.retryListener.onRetryPolicyInterruption(this.retryPolicy, retryable, retryException);
                    throw retryException;
                }
                logger.debug(() -> "Preparing to retry operation '%s'".formatted(retryableName));
                try {
                    this.retryListener.beforeRetry(this.retryPolicy, retryable);
                    R result = retryable.execute();
                    this.retryListener.onRetrySuccess(this.retryPolicy, retryable, result);
                    logger.debug(() -> "Retryable operation '%s' completed successfully after retry".formatted(retryableName));
                    return result;
                }
                catch (Throwable currentException) {
                    logger.debug(currentException, () -> "Retry attempt for operation '%s' failed due to '%s'".formatted(retryableName, currentException));
                    this.retryListener.onRetryFailure(this.retryPolicy, retryable, currentException);
                    exceptions.add(currentException);
                    lastException = currentException;
                }
            }
            RetryException retryException = new RetryException("Retry policy for operation '%s' exhausted; aborting execution".formatted(retryableName), (Throwable)exceptions.removeLast());
            exceptions.forEach(retryException::addSuppressed);
            this.retryListener.onRetryPolicyExhaustion(this.retryPolicy, retryable, retryException);
            throw retryException;
        }
    }

    private static class RetryInterruptedException
    extends RetryException {
        private static final long serialVersionUID = 1L;

        RetryInterruptedException(String message, InterruptedException cause) {
            super(message, cause);
        }

        @Override
        public int getRetryCount() {
            return this.getSuppressed().length - 1;
        }
    }
}

